<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Auth;

class ForumBoard extends Model
{
    use HasFactory;

    protected $fillable = [
        'category_id',
        'name',
        'slug',
        'description',
        'order',
        'is_active',
        'required_roles',
        'required_permissions',
        'is_public',
    ];

    protected $casts = [
        'required_roles' => 'array',
        'required_permissions' => 'array',
        'is_public' => 'boolean',
    ];

    public function category(): BelongsTo
    {
        return $this->belongsTo(ForumCategory::class, 'category_id');
    }

    public function threads(): HasMany
    {
        return $this->hasMany(ForumThread::class, 'board_id');
    }

    public function latestThreads()
    {
        return $this->threads()->latest()->take(5);
    }
    
    public function threadCount()
    {
        return $this->threads()->count();
    }
    
    public function postCount()
    {
        return ForumPost::whereHas('thread', function ($query) {
            $query->where('board_id', $this->id);
        })->count();
    }

    /**
     * Check if the current user can access this board
     */
    public function canAccess($user = null): bool
    {
        $user = $user ?: Auth::user();

        // First check if user can access the parent category
        if (!$this->category->canAccess($user)) {
            return false;
        }

        // If board is public, inherit category access
        if ($this->is_public) {
            return true;
        }

        // If no user is logged in and board is not public
        if (!$user) {
            return false;
        }

        // Check required roles
        if (!empty($this->required_roles)) {
            $hasRequiredRole = false;
            foreach ($this->required_roles as $role) {
                if ($user->hasRole($role)) {
                    $hasRequiredRole = true;
                    break;
                }
            }
            if (!$hasRequiredRole) {
                return false;
            }
        }

        // Check required permissions
        if (!empty($this->required_permissions)) {
            foreach ($this->required_permissions as $permission) {
                if (!$user->can($permission)) {
                    return false;
                }
            }
        }

        return true;
    }

    /**
     * Scope to filter boards accessible by current user
     */
    public function scopeAccessible(Builder $query)
    {
        $user = Auth::user();

        return $query->whereHas('category', function ($categoryQuery) {
            $categoryQuery->accessible();
        })->where(function ($q) use ($user) {
            // Always include public boards
            $q->where('is_public', true);

            if ($user) {
                // Include boards based on roles
                $q->orWhere(function ($roleQuery) use ($user) {
                    $roleQuery->whereJsonContains('required_roles', $user->getRoleNames()->toArray())
                             ->orWhereNull('required_roles');
                });

                // Include boards based on permissions
                $q->orWhere(function ($permQuery) use ($user) {
                    $userPermissions = $user->getAllPermissions()->pluck('name')->toArray();
                    if (!empty($userPermissions)) {
                        foreach ($userPermissions as $permission) {
                            $permQuery->orWhereJsonContains('required_permissions', $permission);
                        }
                    }
                });
            }
        });
    }
}